﻿namespace Hims.Api.Controllers
{
    using System;
    using System.Collections.Generic;
    using System.Linq;
    using System.Threading.Tasks;
    using Domain.Services;
    using Hims.Api.Models;
    using Microsoft.AspNetCore.Authorization;
    using Microsoft.AspNetCore.Mvc;
    using Microsoft.CodeAnalysis.CSharp;
    using Npgsql;
    using Shared.DataFilters;
    using Shared.EntityModels;
    using Shared.Library.Enums;
    using Shared.UserModels.Filters;
    using Utilities;

    /// <inheritdoc />
    [Authorize]
    [Route("api/floors")]
    [Consumes("application/json")]
    [Produces("application/json")]
    public class FloorController : BaseController
    {
        /// <summary>
        /// The coupon services.
        /// </summary>
        private readonly IFloorService floorService;

        /// <summary>
        /// The audit log services.
        /// </summary>
        private readonly IAuditLogService auditLogServices;

        /// <inheritdoc />
        public FloorController(IFloorService floorService, IAuditLogService auditLogServices)
        {
            this.floorService = floorService;
            this.auditLogServices = auditLogServices;
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("fetch")]
        [ProducesResponseType(typeof(List<FloorModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchAsync([FromBody]FloorFilterModel model)
        {
            model = (FloorFilterModel)EmptyFilter.Handler(model);          
            model.LocationId = model.LocationId >0? model.LocationId :null;
            var floors = await this.floorService.FetchAsync(model);
            return floors == null ? this.ServerError() : this.Success(floors);
        }

        /// <summary>
        /// The fetch coupons.
        /// </summary>
        /// <param name="model">
        /// The coupon filter model.
        /// </param>
        /// <returns>
        /// The list of coupons.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - List of coupons.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpGet]
        [Route("floor-fetch")]
        [ProducesResponseType(typeof(List<FloorModel>), 200)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> FetchFloorAsync([FromHeader] LocationHeader header)
        {
            var floors = await this.floorService.FetchFloorAsync();
            return Success(floors);
        }


        /// <summary>
        /// The add coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon added successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("add")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> AddAsync([FromBody] FloorModel model,[FromHeader] LocationHeader header)
        {
            model = (FloorModel)EmptyFilter.Handler(model);
            model.LocationId = model.LocationId ==null ? int.Parse(header.LocationId) : model.LocationId;
            var response = await this.floorService.AddAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given floor code has already been exists at given Location.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.CreatedBy,
                LogTypeId = (int)LogTypes.Floor,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.CreatedByName } has added Floor : <b>{model.FloorName}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Floor has been added successfully.");
        }

        /// <summary>
        /// The update coupon.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Coupon updated successfully.
        /// - 409 - Coupon already exist.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPut]
        [Route("update")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> UpdateAsync([FromBody] FloorModel model,[FromHeader] LocationHeader header)
        {
            model = (FloorModel)EmptyFilter.Handler(model);
            model.LocationId = model.LocationId == null ? int.Parse(header.LocationId) : model.LocationId;
            var response = await this.floorService.UpdateAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Given Floor has already been exists at given Location.");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Floor,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has updated Floor : <b>{model.FloorName}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Floor has been updated successfully.");
        }

        /// <summary>
        /// The delete specialization.
        /// </summary>
        /// <param name="model">
        /// The model.
        /// </param>
        /// <returns>
        /// The <see cref="Task"/>.
        /// </returns>
        /// <remarks>
        /// ### REMARKS ###
        /// The following codes are returned
        /// - 200 - Specialization deleted successfully.
        /// - 409 - Specialization can not be deleted.
        /// - 500 - Problem with Server side code.
        /// </remarks>
        [HttpPost]
        [Route("delete")]
       // [ProducesResponseType(typeof(string), 200)]
       // [ProducesResponseType(409)]
       // [ProducesResponseType(500)]
        public async Task<ActionResult> DeleteAsync([FromBody] FloorModel model, [FromHeader] LocationHeader header)
        {
           
                model = (FloorModel)EmptyFilter.Handler(model);
                var response = await this.floorService.DeleteAsync(model.FloorId);

            switch (response)
            {
                case -1:
                    return this.Conflict("Unable to Find Floor");
                case -2:
                    return this.Conflict("Floor's Bed is allocated, please Deallocate then try");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Floor,
                LogFrom = (short)model.LoginRoleId,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" { model.ModifiedByName } has deleted Floor - <b>{model.FloorName}</b> on {DateTime.UtcNow.AddMinutes(330).ToString("dd/MM/yyyy, hh:mm tt")}",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Floor has been deleted successfully.");
           
        }

        /// <summary>
        /// Changes the status asynchronous.
        /// </summary>
        /// <param name="model">The model.</param>
        /// <param name="header">The header.</param>
        /// <returns></returns>
        [HttpPut]
        [Route("change-status")]
        [ProducesResponseType(typeof(string), 200)]
        [ProducesResponseType(409)]
        [ProducesResponseType(500)]
        public async Task<ActionResult> ChangeStatusAsync([FromBody] FloorModel model, [FromHeader] LocationHeader header)
        {
            model = (FloorModel)EmptyFilter.Handler(model);
            var response = await this.floorService.ChangeStatusAsync(model);
            switch (response)
            {
                case -1:
                    return this.Conflict("Unable to Find Floor");
                case -2:
                    return this.Conflict("Floor's Bed is allocated, please Deallocate then try");
                case 0:
                    return this.ServerError();
            }

            var auditLogModel = new AuditLogModel
            {
                AccountId = model.ModifiedBy,
                LogTypeId = (int)LogTypes.Floor,
                LogFrom = (int)AccountType.Administrator,
                LogDate = DateTime.UtcNow.AddMinutes(330),
                LogDescription = $" {model.ModifiedByName} has changed status of  Floor - {model.FloorName}  to {(model.Active ? "Active" : "Deactivate")  } on { DateTime.UtcNow.AddMinutes(330) }",
                LocationId = Convert.ToInt32(header.LocationId)
            };
            await this.auditLogServices.LogAsync(auditLogModel);

            return this.Success("Floor Status has been updated successfully.");
        }

    }

}